from django.db import models
from .UserProfile import UserProfile
from django.utils.text import slugify


class Volunteer(models.Model):
    VOLUNTEER_TYPES = [
        ('VIRTUAL', 'Remote Volunteer Opportunity'),
        ('LONG_TERM', 'Long-Term Volunteer Commitment'),
        ('SHORT_TERM', 'Short-Term Volunteer Commitment'),
        ('SPECIALIZED', 'Specialized Skill-Based Volunteer'),
        ('EVENT_BASED', 'Event-Based Volunteer Participation'),  
        ('MENTORING', 'Mentorship and Guidance Volunteer'),      
        ('ADVOCACY', 'Advocacy and Awareness Volunteer'),        
        ('ADMINISTRATIVE', 'Administrative Support Volunteer'),  
        ('COMMUNITY_SERVICE', 'Community Service Volunteer'),    
        ('RESEARCH', 'Research and Data Analysis Volunteer'),    
    ]
    
    profile_picture = models.ImageField(upload_to='Volunteer profile_pictures/', blank=True)
    name = models.CharField(max_length=100, null=True, blank=True)
    email = models.EmailField(null=True, blank=True)
    phone = models.CharField(max_length=20)
    volunteer_type = models.CharField(max_length=100, choices=VOLUNTEER_TYPES, default='COMMUNITY_SERVICE', help_text='Select the type of volunteer')
    
    start_date = models.DateField(null=True, blank=True)
    end_date = models.DateField(null=True, blank=True)
    
    licenses = models.ManyToManyField('SupportDocument', related_name="volunteer_licenses", blank=True, help_text='Volunteer Licenses')
    role_description = models.TextField(max_length=500, null=True, blank=True)
    additional_info = models.TextField(max_length=500, blank=True, help_text="Provide any relevant information about your capacities, special needs, or other details.")
    
    emergency_contact_name = models.CharField(max_length=100, null=True, blank=True)
    emergency_contact_phone = models.CharField(max_length=20,null=True, blank=True)
    emergency_contact_email = models.EmailField(null=True, blank=True)
    
    date_applied = models.DateTimeField(auto_now_add=True)
    is_active = models.BooleanField(default=False)
    volunteer_agreement_accepted = models.BooleanField(default=False)
    slug = models.SlugField(max_length=255, editable=False, unique=True, blank=True)

    def save(self, *args, **kwargs):
        if not self.slug:
            base_slug = slugify(self.name + '-' + self.email)
            slug = base_slug
            counter = 1
            while Volunteer.objects.filter(slug=slug).exists():
                slug = f"{base_slug}-{counter}"
                counter += 1
            self.slug = slug
        super().save(*args, **kwargs)

    def __str__(self):
        return f"Volunteer: {self.name} {self.email}"
