import uuid
import random
import string
from django.db import models
from django.utils.text import slugify
from django.contrib.auth.models import User



def generate_verification_token():
    return ''.join(random.choice(string.ascii_letters + string.digits) for _ in range(20))

class UserProfile(models.Model):
    user = models.OneToOneField(User, on_delete=models.CASCADE, help_text="A one-to-one relationship with the default Logged in User. Example: 'You!'" )
    profile_picture = models.ImageField(upload_to='profile_pictures/', blank=True)
    phone = models.CharField(max_length=20)
    # countries eastern africa
    bio = models.TextField(blank=True)
    slug = models.SlugField(unique=True, editable=False, blank=True)
    verification_token = models.CharField(max_length=20, default=generate_verification_token, unique=False, blank=True)
    created_at = models.DateTimeField(auto_now_add=True)
    is_active = models.BooleanField(default=True)
    is_verified = models.BooleanField(default=True)
    resume_and_certifications = models.ManyToManyField('SupportDocument', blank=True, help_text='Upload resumes, certificates, awards..etc')
    
    is_nurse = models.BooleanField(default=False)
    is_doctor = models.BooleanField(default=False)
    is_pharmacist = models.BooleanField(default=False)
    is_specialist = models.BooleanField(default=False)
    is_partner = models.BooleanField(default=False)
    is_unknown = models.BooleanField(default=False)
    is_volunteer = models.BooleanField(default=False)
    is_member = models.BooleanField(default=False)
    is_finance_officer = models.BooleanField(default=False)
    is_manager = models.BooleanField(default=False)
    is_board_member = models.BooleanField(default=False)
    
    def __str__(self):
        return f"{self.user.first_name} {self.user.last_name} ({self.user.email})"
    
    def save(self, *args, **kwargs):
        # Set unique slug if not already set
        if not self.slug:
            unique_identifier = uuid.uuid4().hex[:8]
            self.slug = slugify(f"{self.user.first_name}-{unique_identifier}")
        
        # Ensure mutual exclusivity of roles
        if self.is_unknown:
            self.is_nurse = False
            self.is_doctor = False
            self.is_pharmacist = False
            self.is_partner = False
            self.is_moderator = False
        else:
            # If any other role is set, is_unknown must be False
            if self.is_nurse or self.is_doctor or self.is_pharmacist or self.is_partner or self.is_moderator:
                self.is_unknown = False

        super(UserProfile, self).save(*args, **kwargs)
