from decimal import Decimal
from django.apps import apps
from django.db import models
from django.urls import reverse
from django.dispatch import receiver
from django.utils.text import slugify
from django.db.models.signals import post_save
from django_ckeditor_5.fields import CKEditor5Field
from django.db import IntegrityError, transaction

from .UserProfile import UserProfile
from .calendar import Calendar
from .PartneringOrganization import PartneringOrganization


# Indicator model
class Indicator(models.Model):
    MEASURE_TYPE_CHOICES = [
        ('Quantitative', 'Quantitative: Measures that are numeric and can be counted or measured. Examples include percentages, ratios, and rates.'),
        ('Qualitative', 'Qualitative: Measures that describe qualities or characteristics. Examples include levels of satisfaction, opinions, and perceptions.'),
    ]

    MEASURE_UNIT_CHOICES = [
        ('Percentage', 'Percentage: Represents a portion out of 100. Example: 75% satisfaction.'),
        ('Rate', 'Rate: A measure of how frequently something occurs. Example: 5 events per month.'),
        ('Ratio', 'Ratio: A quantitative relationship between two numbers. Example: 3:1 ratio of success to failure.'),
        ('Opinion', 'Opinion: A subjective measure reflecting personal views or beliefs. Example: Survey responses about preferences.'),
        ('Perception', 'Perception: A qualitative measure of how people perceive or view something. Example: Public perception of a service.'),
        ('Level of Satisfaction', 'Level of Satisfaction: A qualitative measure of how satisfied individuals are, often using scales. Example: High, Medium, Low.'),
    ]
    
    DATA_SOURCE_CHOICES = [
    ('Survey', 'Survey: Data collected through surveys or questionnaires.'),
    ('Reports', 'Reports: Data derived from published reports or studies.'),
    ('Focus Groups', 'Focus Groups: Data collected from group discussions or focus groups.'),
    ('Case Studies', 'Case Studies: Detailed examinations of specific instances or examples.'),
    ('Interviews', 'Interviews: Data collected through structured or semi-structured interviews.'),
    ('Experimental Data', 'Experimental Data: Data derived from controlled experiments or trials.'),
    ('Statistical Data', 'Statistical Data: Data collected and analyzed using statistical methods.'),
    ('Benchmarking', 'Benchmarking: Data obtained by comparing against standards or best practices.'),
    ('Public Records', 'Public Records: Data obtained from publicly available records or documents.'),
    ('Field Observations', 'Field Observations: Data collected through direct observations in the field.'),
    ('Expert Assessments', 'Expert Assessments: Data gathered from evaluations by subject matter experts.'),
    ('Administrative Data', 'Administrative Data: Data collected by government or other administrative bodies.'),
    ('Digital Analytics', 'Digital Analytics: Data collected from digital platforms, such as website or app usage.'),
    ('Other', 'Other: Any other sources not listed here.'),
    ]

    name = models.CharField( max_length=255,  help_text="Enter the name of the indicator. Example: 'Increase in literacy rate'." )
    description = models.TextField( help_text="Provide a detailed description of the indicator. Example: 'This indicator measures the percentage increase in literacy among adults.'" )
    measure_type = models.CharField( max_length=15,  choices=MEASURE_TYPE_CHOICES,  help_text="Choose whether the measure is quantitative or qualitative." )
    measure_unit = models.CharField( max_length=50,  choices=MEASURE_UNIT_CHOICES,  help_text="Choose the unit of measure for this indicator. Example: '%', 'Ratio', 'Level of Satisfaction', etc." )
    data_source = models.CharField(max_length=50, choices=DATA_SOURCE_CHOICES, null=True, blank=True, help_text="Specify the source of data used for this indicator. Example: 'Survey', 'Administrative Data'.")
    baseline = models.CharField(max_length=255, null=True, blank=True, help_text="Enter the baseline value before the intervention. For quantitative, example: '50%' for a literacy rate; for qualitative, example: 'Low satisfaction'.")
    target = models.CharField(max_length=255, null=True, blank=True, help_text="Enter the target value that is hoped to be achieved. For quantitative, example: '75%' for a literacy rate; for qualitative, example: 'High satisfaction'.")
    is_target_achieved = models.BooleanField(default=False, help_text="Indicate whether the target for this indicator has been achieved. Example: 'True' if the target value has been met or exceeded.")
    achieved_value = models.CharField(max_length=255, null=True, blank=True, help_text="Enter the achieved value. For quantitative, example: '70%' for a literacy rate; for qualitative, example: 'Moderate satisfaction'.")
    notes = models.TextField(null=True, blank=True, help_text="Describe any lessons learned from the process of achieving (or not achieving) this indicator. Example: 'Need to increase community engagement for better outcomes.'")
    relevance = models.TextField( help_text="Explain how this indicator is relevant to the subject and evaluation. Example: 'Relevant to assessing the impact of education programs on literacy.'" )
    adequacy = models.TextField( help_text="Describe how this indicator provides sufficient information on performance. Example: 'Adequate to determine the effectiveness of the literacy program.'" )
    economic = models.BooleanField( default=True,  help_text="Indicate whether this indicator can be measured at a reasonable cost. Example: 'True' for an indicator that uses readily available data." )
    monitorable = models.BooleanField( default=True,  help_text="Indicate whether this indicator is easily monitored and amenable to independent validation. Example: 'True' if the data can be independently verified." )
    slug = models.SlugField(max_length=255, unique=True, editable=False, blank=True)
    
    def save(self, *args, **kwargs):
        if not self.slug:
            base_slug = slugify(self.name)
            slug = base_slug
            counter = 1
            while Indicator.objects.filter(slug=slug).exists():
                slug = f"{base_slug}-{counter}"
                counter += 1
            self.slug = slug
        super(Indicator, self).save(*args, **kwargs)
        
    def __str__(self):
        return self.name

    class Meta:
        verbose_name = 'Project Indicator'
        verbose_name_plural = 'Project Indicators'

# Beneficiary model
class Beneficiary(models.Model):
    CATEGORY_CHOICES = [
        ('Men', 'Men'),
        ('Women', 'Women'),
        ('Children', 'Children'),
        ('Disabled', 'Disabled'),
        ('Youth', 'Youth'),
        ('girls', 'girls'),
        ('boys', 'boys'),
        ('community', 'community'),
        ('employed', 'employed'),
        ('unemployed', 'unemployed'),
        ('students', 'students'),
        ('entreprenurs', 'entrepreneurs'),
        ('farmers', 'farmers'),
        ('artisans', 'artisans'),
        ('Elderly', 'Elderly'),
        ('Refugees', 'Refugees'),
        ('Indigenous', 'Indigenous'),
        ('Homeless', 'Homeless'),
        ('Other', 'Other'),
    ]

    PRIORITY_CHOICES = [
        ('High', 'High'),
        ('Medium', 'Medium'),
        ('Low', 'Low'),
        ('Critical', 'Critical'),
        ('Urgent', 'Urgent'),
        ('Normal', 'Normal'),
    ]
    
    name = models.CharField(max_length=255, help_text="Enter the full name of the beneficiary eg John Doe, Entebe primary school, Family childrens home, etc.")
    location = models.CharField(max_length=255, help_text="Specify the geographical location of the beneficiaries. Example: 'Nairobi, Kenya, Kiserian, Kiambu, Loitoktok'.")
    description = models.TextField(help_text="Provide additional details about the beneficiary. Example: 'A group of 25 young girls from Nunu primary school in Entebe rural area with limited access to education.'")
    category = models.CharField(max_length=50, choices=CATEGORY_CHOICES, help_text="Choose the category that best describes the beneficiary group.")
    priority = models.CharField(max_length=10, choices=PRIORITY_CHOICES, help_text="Choose the priority level assigned to the beneficiary group.")
    strength = models.TextField(help_text="Describe the strengths of the beneficiary group. Example: 'Strong community leadership.'")
    weakness = models.TextField(help_text="Describe the weaknesses or challenges faced by the beneficiary group. Example: 'Limited financial resources.'")
    opportunities = models.TextField(help_text="List opportunities available to the beneficiary group. Example: 'Access to government grants.'")
    threats = models.TextField(help_text="List threats or risks faced by the beneficiary group. Example: 'Political instability in the region.'")
    needs = models.TextField(help_text="Outline the needs of the beneficiary group. Example: 'Access to healthcare services.'")
    when_to_engage = models.TextField(help_text="Specify the appropriate times or phases during the project when the beneficiary group should be engaged. Example: 'During the initial planning phase.'")
    how_to_engage = models.TextField(help_text="Describe the methods or approaches for engaging with the beneficiary group. Example: 'Community meetings and focus group discussions.'")
    engagement_channels = models.TextField(help_text="List the communication channels to be used for engaging with the beneficiary group. Example: 'Email, SMS, in-person meetings.'")
    engagement_frequency = models.TextField(help_text="Specify how often the beneficiary group will be engaged. Example: 'Monthly check-ins.'")
    support_document = models.ManyToManyField('SupportDocument', blank=True, help_text='Upload documents to verify the cash received eg receipt, bank statement, etc')
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)
    slug = models.SlugField(max_length=255, unique=True, editable=False, blank=True)
    
    def save(self, *args, **kwargs):
        if not self.slug:
            base_slug = slugify(self.name)
            slug = base_slug
            counter = 1
            while Beneficiary.objects.filter(slug=slug).exists():
                slug = f"{base_slug}-{counter}"
                counter += 1
            self.slug = slug
        super(Beneficiary, self).save(*args, **kwargs)
        
    def __str__(self):
        return self.name
    
    class Meta:
        verbose_name = 'Project Beneficiary'
        verbose_name_plural = 'Project Beneficiaries'

# Stakeholder model
class Stakeholder(models.Model):
    
    AFFECT_PROJECT_CHOICES = [
        (1, 'No impact'),
        (2, 'Minimal impact'),
        (3, 'Moderate impact'),
        (4, 'Significant impact'),
        (5, 'Critical impact'),
    ]

    AFFECTED_BY_PROJECT_CHOICES = [
        (1, 'Not affected'),
        (2, 'Slightly affected'),
        (3, 'Moderately affected'),
        (4, 'Significantly affected'),
        (5, 'Highly affected'),
    ]

    AFFECT_PROJECT_RESULTS_CHOICES = [
        (1, 'No influence on results'),
        (2, 'Minor influence on results'),
        (3, 'Moderate influence on results'),
        (4, 'Strong influence on results'),
        (5, 'Decisive influence on results'),
    ]

    AFFECTED_BY_PROJECT_RESULTS_CHOICES = [
        (1, 'No impact by results'),
        (2, 'Minimal impact by results'),
        (3, 'Moderate impact by results'),
        (4, 'Significant impact by results'),
        (5, 'Critical impact by results'),
    ]


    orgnanization = models.ForeignKey('PartneringOrganization', on_delete=models.CASCADE, null=True, blank=True, help_text="Choose the partnering organization of the stakeholder.")
    name = models.CharField(max_length=255, help_text="Enter the full name of the stakeholder eg John Doe, Entebe primary school, Family childrens home, etc.")
    location = models.CharField(max_length=255, help_text="Specify the geographical location of the stakeholders. Example: 'Nairobi, Kenya, Kiserian, Kiambu, Loitoktok'.")
    description = models.TextField(help_text="Provide additional details about the stakeholder. Example: 'Jane is a community leader involved in youth empowerment initiatives.'")
    strength = models.TextField(help_text="Describe the strengths of the stakeholder. Example: 'Strong networking skills and community influence.'")
    contribution = models.TextField(help_text="Describe the contribution of the stakeholder to the project. Example: 'Providing Networking support and community mobilization.'")
    expectations = models.TextField(help_text="List expectations of the stakeholder. Example: 'Timely project updates and involvement in decision-making processes.'")
    interests = models.TextField(help_text="Outline the interests and goals of the stakeholder. Example: 'Improving educational opportunities for underprivileged children.'")
    affect_project = models.IntegerField( choices=AFFECT_PROJECT_CHOICES, help_text="Rate how much the stakeholder influences the overall project. " "1 = No impact, 5 = Critical impact." )
    affected_by_project = models.IntegerField( choices=AFFECTED_BY_PROJECT_CHOICES, help_text="Rate how much the stakeholder is affected by the project's activities. " "1 = Not affected, 5 = Highly affected." )
    affect_project_results = models.IntegerField( choices=AFFECT_PROJECT_RESULTS_CHOICES, help_text="Rate how much the stakeholder influences the results or outcomes of the project. " "1 = No influence on results, 5 = Decisive influence on results." )
    affected_by_project_results = models.IntegerField( choices=AFFECTED_BY_PROJECT_RESULTS_CHOICES, help_text="Rate how much the stakeholder is affected by the results or outcomes of the project. " "1 = No impact by results, 5 = Critical impact by results." )
    gain_from_project = models.TextField( help_text="Describe what the stakeholder gains or benefits from the project." )
    support_document = models.ManyToManyField('SupportDocument', blank=True, help_text='Upload documents to verify the cash received eg receipt, bank statement, etc')
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)
    slug = models.SlugField(max_length=255, unique=True, editable=False, blank=True)
    
    def save(self, *args, **kwargs):
        if not self.slug:
            base_slug = slugify(self.name)
            slug = base_slug
            counter = 1
            while Stakeholder.objects.filter(slug=slug).exists():
                slug = f"{base_slug}-{counter}"
                counter += 1
            self.slug = slug
        super(Stakeholder, self).save(*args, **kwargs)
        
    def __str__(self):
        return self.name
    
    class Meta:
        verbose_name = 'Project Stakeholder'
        verbose_name_plural = 'Project Stakeholders'

# Project model
class Project(models.Model):
    
    CATEGORY_CHOICES = [
        ('Proprietary', 'Proprietary'),
        ('Collaborative', 'Collaborative'),
    ]
    STATUS_CHOICES = [
        ('Pending', 'Pending'),
        ('Completed', 'Completed'),
        ('In Progress', 'In Progress'),
        ('Not Started', 'Not Started'),
    ]
    PROJECT_CYCLE_CHOICES = [
        ('Design', 'Design'),
        ('Closure', 'Closure'),
        ('Control', 'Control'),
        ('Reporting', 'Reporting'),
        ('Monitoring', 'Monitoring'),
        ('Completion', 'Completion'),
        ('Evaluation', 'Evaluation'),
        ('Dissemination', 'Dissemination'),
        ('Identification', 'Identification'),
        ('Implementation', 'Implementation'),
        ('Pre-implementation', 'Pre-implementation'),
        ('Operational Planning and Budgeting', 'Operational Planning and Budgeting'),
    ]
    PROJECT_PHASE_CHOICES = [
        ('Research', 'Research'),
        ('Engagement', 'Engagement'),
        ('Analysis', 'Analysis'),
        ('Field Work', 'Field Work'),
        ('Planning', 'Planning'),
        ('Execution', 'Execution'),
        ('Review', 'Review'),
        ]

    
    author = models.ForeignKey(UserProfile, on_delete=models.CASCADE, related_name='Project')
    title = models.CharField(max_length=255 , help_text="What is the Title of the project?")
    description = CKEditor5Field('Detailed Description', config_name='extends', help_text="What is the Project About?")
    category = models.CharField(max_length=20, choices=CATEGORY_CHOICES, default='Collaborative', unique=True)
    status = models.CharField(max_length=100, choices=STATUS_CHOICES, default='Not Started')
    stakeholders = models.ManyToManyField(Stakeholder, blank=True, help_text="Who are the Stakeholders of the project?")
    beneficiaries = models.ManyToManyField(Beneficiary, blank=True, help_text="Who are the Beneficiaries of the project?")
    partners = models.ManyToManyField(PartneringOrganization, blank=True, help_text="Who are the Partnering Organization in this project?")
    donation_needed = models.DecimalField(max_digits=15, decimal_places=2, null=True, blank=True)
    donation_raised = models.DecimalField(max_digits=15, decimal_places=2, null=True, blank=True)
    project_cycle = models.CharField( max_length=100, choices=PROJECT_CYCLE_CHOICES, help_text="What is the Current project cycle stage ?." ) 
    project_phase = models.CharField( max_length=100, choices=PROJECT_PHASE_CHOICES, help_text="What is the Current project phase?." )
    activities = models.ManyToManyField ('Activity', blank=True, help_text="Choose the activities of this project.")
    problem_statement = CKEditor5Field('Problem Statement', config_name='extends', blank=True, null=True, help_text="What are the Challanges and problem being faced ?.")
    solution_trees = models.ManyToManyField('SolutionTree', blank=True, help_text="Include solution trees related to this project.")
    problem_impact = CKEditor5Field('Impact of the Problem', config_name='extends', blank=True, null=True, help_text="What are the Negative impacts and effects impacts of the project to environment and beneficiaries.?")
    project_goals = CKEditor5Field('Projected goals', config_name='extends', blank=True, null=True, help_text="What are the goals of the project ?.")
    sustainability_plan = CKEditor5Field( 'Sustainability Plan', config_name='extends', blank=True, null=True, help_text="What measures are in place to ensure the long-term sustainability and impact of the project?" )
    views_count = models.PositiveIntegerField(default=0)
    views_ips = models.JSONField(default=list, blank=True)
    project_document = models.ManyToManyField('SupportDocument', blank=True, help_text='Upload documents related to the project initiative THAT DO NOT NEED UPDATE eg Last-report, analysis, awards, etc')
    is_active = models.BooleanField(default=False)
    updated_at = models.DateTimeField(auto_now=True)
    created_at = models.DateTimeField(auto_now_add=True)
    slug = models.SlugField(unique=True, max_length=500, editable=False, blank=True)
    

    def __str__(self):
        return f"{self.title} -Project"
    
    class Meta:
        verbose_name = 'Project'
        verbose_name_plural = 'Project'

    def save(self, *args, **kwargs):
        if not self.slug:
            base_slug = slugify(self.title)
            slug = base_slug
            counter = 1
            while Project.objects.filter(slug=slug).exists():
                slug = f"{base_slug}-{counter}"
                counter += 1
            self.slug = slug
        super(Project, self).save(*args, **kwargs)

    
    
    @property
    def donation_remaining(self):
        
        donation_needed = self.donation_needed or Decimal('0')
        donation_raised = self.donation_raised or Decimal('0')
        return max(Decimal('0'), donation_needed - donation_raised)
        

# Resource models
class Resource(models.Model):
    name = models.CharField(max_length=255, help_text="Enter the name of the resource. Example: 'Projector'.")
    number = models.IntegerField(help_text="Specify the quantity of the resource required. Example: '2'.")
    unit_cost = models.DecimalField(max_digits=10, decimal_places=2, help_text="Enter the unit cost of the resource. Example: '500.00'.")
    total_cost = models.DecimalField(max_digits=15, decimal_places=2, help_text="This field is auto-calculated based on the number and unit cost.")
    use_function = models.TextField(help_text="Describe the use or function of the resource within the project. Example: 'Used for presentations during training sessions.'")
    slug = models.SlugField(max_length=255, unique=True, editable=False, blank=True)
    
    def save(self, *args, **kwargs):
        if not self.slug:
            base_slug = slugify(self.name)
            slug = base_slug
            counter = 1
            while Resource.objects.filter(slug=slug).exists():
                slug = f"{base_slug}-{counter}"
                counter += 1
            self.slug = slug
        self.total_cost = self.number * self.unit_cost
        super(Resource, self).save(*args, **kwargs)
        

    def __str__(self):
        return f"{self.name} (x{self.number})"

    class Meta:
        verbose_name = 'Project Resource'
        verbose_name_plural = 'Project Resources'

# Activity model
class Activity(models.Model):
    STATUS_CHOICES = [
        ('Planned', 'Planned'),
        ('In Progress', 'In Progress'),
        ('Completed', 'Completed'),
        ('Delayed', 'Delayed'),
        ('Canceled', 'Canceled'),
        ('On Hold', 'On Hold'),
        ('Not Started', 'Not Started'),
        ('Reviewed', 'Reviewed'),
        ('Approved', 'Approved'),
    ]

    name = models.CharField(max_length=255, help_text="Enter the name of the activity. Example: 'Community Training on Health Education'.")
    resources = models.ManyToManyField(Resource, help_text="Choose and allocate all resources required for this activity.")
    stakeholders = models.ManyToManyField('Stakeholder', help_text="Choose stakeholders involved in this activity.")
    beneficiaries = models.ManyToManyField('Beneficiary', help_text="Choose beneficiaries of this activity.")
    status = models.CharField(max_length=20, choices=STATUS_CHOICES, help_text="Choose the current status of the activity.")
    start_date = models.DateField(help_text="Specify the start date of the activity. Example: '2024-08-01'.")
    end_date = models.DateField(help_text="Specify the end date of the activity. Example: '2024-08-05'.")
    goals_and_objectives = models.TextField(help_text="Define the goals and objectives of this activity. Example: 'Goal is to enhance community health literacy and objective is to train 100 community members on basic health practices.'" )
    user_responsible = models.ForeignKey('UserProfile', on_delete=models.SET_NULL, null=True, blank=True, help_text="Choose the user responsible for overseeing the implementation of this activity.")
    inputs_needed = models.TextField(help_text="List the inputs required to carry out this activity. Example: 'Trainers, training materials, venue, equipment.'")
    outputs_needed_to_achieve_outcomes = models.TextField(help_text="Specify the outputs needed to achieve the desired outcomes. Example: 'Trained community members, distributed health education materials.'")
    outcomes_to_be_achieved = models.TextField(help_text="Describe the outcomes that this activity aims to achieve. Example: 'Improved health awareness among community members.'")
    impact = models.TextField(help_text="Describe the long-term impact of this activity. Example: 'Reduction in preventable diseases in the community.'")
    strategy = models.TextField(help_text="Outline the strategy for implementing this activity. Example: 'Collaborate with local health professionals and community leaders.'")
    how_activity_will_be_carried_out = models.TextField(help_text="Explain how the activity will be executed. Example: 'Conduct workshops in community centers with the help of trained facilitators.'")
    cost = models.DecimalField(max_digits=15, decimal_places=2, help_text="Enter the total cost of the activity. Example: '1500.00'.")
    support_document = models.ManyToManyField( 'SupportDocument',  blank=True,  help_text="Upload supporting documents." )
    slug = models.SlugField(max_length=255, unique=True, editable=False, blank=True)
    
    def save(self, *args, **kwargs):
        
        if not self.slug:
            base_slug = slugify(self.name)
            slug = base_slug
            counter = 1
            while Activity.objects.filter(slug=slug).exists():
                slug = f"{base_slug}-{counter}"
                counter += 1
            self.slug = slug

        super().save(*args, **kwargs)
        self.update_calendar_event()

    def update_calendar_event(self):
        description = (
            f"Activity: {self.name}\n"
            f"Status: {self.status}\n"
            f"Start Date: {self.start_date}\n"
            f"End Date: {self.end_date}\n"
            f"Responsible User: {self.user_responsible}\n"
            f"Objectives: {self.goals_and_objectives}\n"
            f"Cost: {self.cost}\n"
        )
        Calendar.objects.update_or_create(
            title=self.name,
            defaults={
                'start': self.start_date,
                'end': self.end_date,
                'color_code': self.get_color_code(),
                'badge': 'Activity',
                'description': description,
            }
        )

    def get_color_code(self):
        if self.status in ['Completed', 'Approved']:
            return 'Success'
        elif self.status in ['In Progress', 'Planned']:
            return 'Info'
        elif self.status in ['Delayed', 'On Hold', 'Canceled']:
            return 'Danger'
        else:
            return 'Primary'

    

    def __str__(self):
        return self.name
    
    def get_absolute_url(self):
        return reverse('edit_activity', kwargs={'slug': self.slug})

    class Meta:
        verbose_name = 'Project Activity'
        verbose_name_plural = 'Project Activities'

# CoreProblem model
class CoreProblem(models.Model):
    name = models.CharField(max_length=255, help_text="Enter the name of the core problem. Example: 'Lack of access to clean water'.")
    description = models.TextField(help_text="Provide a detailed description of the core problem. Example: 'Communities in rural areas have limited access to clean drinking water, leading to health issues.'.")
    driver = models.TextField(help_text="Identify the key drivers contributing to the core problem. Example: 'Poor infrastructure, limited government support'.")
    root_causes = models.TextField(help_text="List the root causes of the core problem. Example: 'Geographic isolation, lack of investment in water systems'.")
    effects = models.TextField(help_text="Describe the effects of the core problem on the community or environment. Example: 'Increased incidence of waterborne diseases'.")
    impact = models.TextField(help_text="Describe the broader impact of the core problem. Example: 'Reduced quality of life, economic strain'.")
    assumption = models.TextField(help_text="List any assumptions related to addressing the core problem. Example: 'Assume that community members are willing to adopt new water practices'.")
    opportunity = models.TextField(help_text="Identify any opportunities that could help address the core problem. Example: 'Potential partnerships with NGOs'.")
    constraints = models.TextField(help_text="List any constraints or barriers to solving the core problem. Example: 'Limited funding, lack of skilled labor'.")
    support_document = models.ManyToManyField('SupportDocument', blank=True, help_text='Upload documents to identify the core problem eg image, video, report  etc')
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)
    slug = models.SlugField(max_length=255, unique=True, editable=False, blank=True)
    
    def save(self, *args, **kwargs):
        if not self.slug:
            base_slug = slugify(self.name)
            slug = base_slug
            counter = 1
            while CoreProblem.objects.filter(slug=slug).exists():
                slug = f"{base_slug}-{counter}"
                counter += 1
            self.slug = slug
        super(CoreProblem, self).save(*args, **kwargs)

    def __str__(self):
        return f"{self.name} {self.description[:50]}"
    
    def get_absolute_url(self):
        return reverse('edit_core_problem', kwargs={'slug': self.slug})
    
    class Meta:
        verbose_name = 'Project Core Problem'
        verbose_name_plural = 'Project Core Problems'
        
# SolutionTree model
class SolutionTree(models.Model):
    core_problem = models.ForeignKey(CoreProblem, on_delete=models.CASCADE, related_name='solution_trees')
    identified_solution = models.TextField(help_text="Describe the identified solution. Example: 'Implementing a water purification system'.")
    solution_drivers = models.TextField(help_text="List the key drivers that support this solution. Example: 'Community demand, availability of resources'.")
    strategy = models.TextField(help_text="Outline the strategy for implementing the solution. Example: 'Partnership with local organizations'.")
    inputs = models.TextField(help_text="Specify the inputs required for the solution. Example: 'Funding, technical expertise'.")
    outputs = models.TextField(help_text="Describe the expected outputs. Example: 'Number of water purification units installed'.")
    outcomes = models.TextField(help_text="Describe the expected outcomes. Example: 'Improved access to clean water'.")
    impact = models.TextField(help_text="Describe the long-term impact of the solution. Example: 'Reduction in waterborne diseases'.")
    sustainability_plan = models.TextField(blank=True, null=True, help_text="Describe the sustainability plan or strategy to ensure long-term impact. Example: 'Training local staff, creating a maintenance fund, community ownership'.")
    expected_results = models.TextField(help_text="Describe the results achieved by the solution. Example: '1000 households with access to clean water'.")
    stakeholders = models.ManyToManyField('Stakeholder', help_text="Choose the stakeholders involved in the solution.")
    beneficiaries = models.ManyToManyField('Beneficiary', help_text="Choose beneficiaries of this Solution.")
    user_responsible = models.ForeignKey('UserProfile', on_delete=models.SET_NULL, null=True, blank=True, help_text="Choose the user responsible for overseeing the implementation of this solution.")
    support_document = models.ManyToManyField( 'SupportDocument',  blank=True,  help_text="Upload supporting documents." )
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)
    slug = models.SlugField(max_length=255, unique=True, editable=False, blank=True)
    
    def save(self, *args, **kwargs):
        if not self.slug:
            base_slug = slugify(f"solution-tree-for-{self.core_problem.name}")
            slug = base_slug
            counter = 1
            while SolutionTree.objects.filter(slug=slug).exists():
                slug = f"{base_slug}-{counter}"
                counter += 1
            self.slug = slug

        super(SolutionTree, self).save(*args, **kwargs)

    
    
    def __str__(self):
        return f"Solution Tree for {self.core_problem.name}"
    
    def get_absolute_url(self):
        return reverse('edit_solution_tree', kwargs={'slug': self.slug})
    
    class Meta:
        verbose_name = 'Project Solution Tree'
        verbose_name_plural = 'Project Solution Trees'
    
# Results model
class Result(models.Model):
    project = models.ForeignKey( 'Project',  on_delete=models.CASCADE, related_name='results',  help_text="Choose the project title associated with this result." )
    output = models.TextField( help_text="Describe the specific output/impact produced by the intervention. Example: '100 community members trained on health practices.'" )
    outcome = models.TextField( help_text="Describe what has changed as a result of the intervention. Example: 'Increased awareness of health practices among trained community members.'" )
    results_achieved = models.TextField( help_text="Specify the results that have been achieved. Example: '80 percent of trained members are now practicing better health hygiene.'" )
    evidence_of_change = models.ManyToManyField( 'Indicator',  help_text="Provide evidence of change/impact by linking relevant indicators, baselines, and targets." )
    
    global_contribution = models.TextField( help_text="Describe how the result contributes to global goals or initiatives. Example: 'Contributes to the United Nations Sustainable Development Goal 3: Good Health and Well-being.'" )
    regional_contribution = models.TextField( help_text="Describe how the result contributes at a regional level. Example: 'Supports regional health initiatives across East Africa.'" )
    national_contribution = models.TextField( help_text="Describe how the result contributes at the national level. Example: 'Aligns with the National Health Strategy of Kenya.'" )
    local_level_contribution = models.TextField( help_text="Describe how the result contributes at the local level. Example: 'Enhances community health and reduces preventable diseases in the locality.'" )
    
    is_achieved = models.BooleanField( default=False,  help_text="Has the result been achieved? Example: 'True' if achieved, 'False' otherwise." )
    gender_inequities_addressed = models.BooleanField( default=False,  help_text="Have gender inequities been addressed in achieving this result? Example: 'True' if addressed, 'False' otherwise." )
    climate_change_addressed = models.BooleanField( default=False,  help_text="Has climate change been addressed in achieving this result? Example: 'True' if addressed, 'False' otherwise." )
    how_gender_inequities_addressed = models.TextField( blank=True,  null=True,  help_text="Explain how gender inequities have been addressed. Example: 'Promoted equal participation of women in training sessions.'" )
    how_climate_change_addressed = models.TextField( blank=True,  null=True,  help_text="Explain how climate change has been addressed. Example: 'Implemented eco-friendly practices during activities.'" )
    support_document = models.ManyToManyField( 'SupportDocument',  blank=True,  help_text="Upload supporting documents that provide evidence of the results achieved." )
    slug = models.SlugField(max_length=255, unique=True, editable=False, blank=True)
    
    def save(self, *args, **kwargs):
        if not self.slug:
            base_slug = slugify(f"Result-for-{self.project.title}")
            slug = base_slug
            counter = 1
            while Result.objects.filter(slug=slug).exists():
                slug = f"{base_slug}-{counter}"
                counter += 1
            self.slug = slug
        super(Result, self).save(*args, **kwargs)
    def __str__(self):
        return f"Result for {self.project.title}"
    
    def get_absolute_url(self):
        return reverse('edit_result', kwargs={'slug': self.slug})

    class Meta:
        verbose_name = 'Project Result'
        verbose_name_plural = 'Project Results'

# Milestone model
class Milestone(models.Model):
    name = models.CharField(max_length=255, help_text="Enter the name of the milestone. Example: 'Completion of Phase 1'.")
    description = models.TextField(blank=True, help_text="Enter a description for the the milestone. Example: 'The project has been completed and all outputs are available.'")
    start_date = models.DateField(help_text="Specify the start date of the milestone.")
    end_date = models.DateField(help_text="Specify the end date of the milestone.")
    project = models.ForeignKey( 'Project',  on_delete=models.CASCADE, related_name='milestones',help_text="Choose the project for this milestone." )
    is_achieved = models.BooleanField(default=False, help_text="Indicate whether this milestone has been achieved. Example: 'True' if achieved, 'False' otherwise.")
    date_of_achievement = models.DateField(blank=True, null=True, help_text="Enter the date when the milestone was achieved. Example: '2024-09-15'.")
    cost = models.DecimalField(max_digits=15, decimal_places=2, help_text="Enter the total cost associated with achieving this milestone. Example: '3000.00'.")
    support_document = models.ManyToManyField( 'SupportDocument',  blank=True,  help_text="Upload supporting documents." )
    slug = models.SlugField(max_length=255, unique=True, editable=False, blank=True)
    
    def save(self, *args, **kwargs):
        if not self.slug:
            base_slug = slugify(f"Project-Milestone-{self.name}-{self.project.title}")
            slug = base_slug
            counter = 1
            while Milestone.objects.filter(slug=slug).exists():
                slug = f"{base_slug}-{counter}"
                counter += 1
            self.slug = slug
        super(Milestone, self).save(*args, **kwargs)
        self.update_calendar_event()

    def update_calendar_event(self):
        description = (
            f"Milestone: {self.name}\n"
            f"Project: {self.project.title}\n"
            f"Start Date: {self.start_date}\n"
            f"End Date: {self.end_date}\n"
            f"Is Achieved: {self.is_achieved}\n"
            f"Date of Achievement: {self.date_of_achievement}\n"
            f"Cost: {self.cost}\n"
        )
        Calendar.objects.update_or_create(
            title=self.name,
            defaults={
                'start': self.start_date,
                'end': self.end_date,
                'color_code': self.get_color_code(),
                'badge': 'Milestone',
                'description': description,
            }
        )

    def get_color_code(self):
        if self.is_achieved:
            return 'Success'
        else:
            return 'Danger'
        
    def __str__(self):
        return self.name
    
    def get_absolute_url(self):
        return reverse('edit_milestone', kwargs={'slug': self.slug})
    class Meta:
        verbose_name = 'Project Milestone'
        verbose_name_plural = 'Project Milestones'
