from datetime import date
from django.db import models
from django.urls import reverse
from django.utils import timezone
from django.utils.text import slugify
from django.contrib.auth.models import User



from .Projects import Project
from .Volunteer import Volunteer
from .UserProfile import UserProfile
from .SupportDocuments import SupportDocument
from .PartneringOrganization import PartneringOrganization


class ProgressReport(models.Model):
    PRIORITY_CHOICES = (
        ('Completed', 'Completed'),
        ('Under-Review', 'Under-Review'),
        ('Not-Started', 'Not-Started'),
    )
    TAG_CHOICES = (
        ('Management', 'Management'),
        ('General','General'),
        ('Finance', 'Finance'),
        ('Program', 'Program'),
    )
    DEL_STATUS = (
        ('trash', 'trash'),
        ('',''),
    )
    STATUS_CHOICES = (
        ('seen', 'Seen'),
        ('trash', 'Trash'),
        ('important', 'Important'),
        ('', 'Default'),
    )
    
    title = models.CharField(max_length=100)
    user_profile = models.ForeignKey(UserProfile, on_delete=models.CASCADE,default=None)
    user_assignee = models.ForeignKey(UserProfile, on_delete=models.CASCADE, null=True, blank=True, related_name='assignee')
    progress_report_priority = models.CharField(max_length=20, choices=PRIORITY_CHOICES, default='Not-Started')
    progress_report_tags = models.CharField(max_length=20, choices=TAG_CHOICES, default='General')
    description = models.TextField(default=0)
    modified_date = models.DateTimeField(auto_now=True)
    created_at = models.DateTimeField(auto_now_add=True)
    seen_by = models.ManyToManyField(UserProfile, related_name='seen_reports', blank=True)
    is_seen = models.BooleanField(default=False)
    project = models.ForeignKey(Project, on_delete=models.CASCADE, null=True, blank=True)
    volunteer_profile = models.ForeignKey(Volunteer, on_delete=models.CASCADE, null=True, blank=True)
    partnering_organization = models.ForeignKey(PartneringOrganization, on_delete=models.CASCADE, null=True, blank=True)
    progress_report_document = models.ManyToManyField(SupportDocument, blank=True)
    slug = models.SlugField(unique=True, editable=False, blank=True)
    class Meta:
        verbose_name = 'Progress Report'
        verbose_name_plural = 'Progress Reports'
        
    def count_seen_by_staff_or_superusers(self):
        return self.seen_by.filter(user__is_staff=True).count() + self.seen_by.filter(user__is_superuser=True).count()
        
    def get_status_for_user(self, user):
        try:
            return UserReportStatus.objects.get(user=user, report=self).status
        except UserReportStatus.DoesNotExist:
            return ''
    
    def __str__(self):
        return f"Progress report by :{self.user_profile.user.first_name} {self.user_profile.user.last_name} ({self.user_profile.user.email})"
        
    def save(self, *args, **kwargs):
        
        if not self.slug:
            base_slug = slugify(self.title)
            slug = base_slug
            counter = 1
            while ProgressReport.objects.filter(slug=slug).exists():
                slug = f"{base_slug}-{counter}"
                counter += 1
            self.slug = slug

        super(ProgressReport, self).save(*args, **kwargs)
        
        
class UserReportStatus(models.Model):
    STATUS_CHOICES = (
        ('seen', 'Seen'),
        ('trash', 'Trash'),
        ('important', 'Important'),
        ('', 'Default'),
    )
    
    user = models.ForeignKey(User, on_delete=models.CASCADE)
    report = models.ForeignKey(ProgressReport, on_delete=models.CASCADE)
    status = models.CharField(max_length=20, choices=STATUS_CHOICES, default='')

    class Meta:
        unique_together = ('user', 'report')

    def __str__(self):
        return f"Status of report {self.report.id} for user {self.user.username}"
    

    
    
    
    
    

