from django.db import models
from django.utils.text import slugify

from .OrganizationType import OrganizationType


class PartneringOrganization(models.Model):
    logo = models.ImageField(upload_to='Partnering_Organizations_logos/')
    name = models.CharField(max_length=100)
    description = models.TextField()
    type = models.ForeignKey(OrganizationType, on_delete=models.SET_DEFAULT, default=None, related_name='partnering_organizations')
    website_link = models.URLField(blank=True, null=True, help_text="Enter the link to the source of the funding opportunity (e.g., official website, announcement page, etc.).")
    
    contact_person_name = models.CharField(max_length=100, null=True, blank=True)
    contact_person_email = models.EmailField(null=True, blank=True)
    contact_person_phone = models.CharField(max_length=20, null=True, blank=True)
    
    role_in_project = models.TextField(blank=True, null=True)
    strategic_importance = models.TextField(blank=True, null=True)
    resource_and_contribution = models.TextField(blank=True, null=True)
    
    letter_of_support = models.ManyToManyField('SupportDocument', related_name="Letter_of_Support", blank=True, help_text='Letters from partnering local NGOs expressing their support and commitment to collaborating on the Project.')
    partnership_document = models.ManyToManyField('SupportDocument', related_name="Partnership_Document", blank=True, help_text='Upload documents to verify the partnership eg M.O.U, etc')
    
    is_verified = models.BooleanField(default=False)
    slug = models.SlugField(unique=True, editable=False, max_length=500, blank=True)
    
    class Meta:
        verbose_name = 'Najimudu Partnering Organization'
        verbose_name_plural = 'Najimudu Partnering Organizations'
    def __str__(self):
        return f"{self.name}"
    
    def save(self, *args, **kwargs):
        if not self.slug:
            base_slug = slugify(self.name)
            slug = base_slug
            counter = 1
            while PartneringOrganization.objects.filter(slug=slug).exists():
                slug = f"{base_slug}-{counter}"
                counter += 1
            self.slug = slug
        super(PartneringOrganization, self).save(*args, **kwargs)
        
        

class PartnerOrganizationFeedbackAndSupport(models.Model):
    partnering_organization = models.ForeignKey(PartneringOrganization, on_delete=models.CASCADE, related_name='feedback_support')
    feedback = models.TextField(blank=True, null=True, help_text="Input from partner organizations about the ongoing project, their experiences, suggestions for improvement, and observations on the project's impact.")
    support = models.TextField(blank=True, null=True, help_text="Input from partner organizations on their experience with the Results Based Management System (RBMS) support, benefits, and challenges faced.")
    date_created = models.DateTimeField(auto_now_add=True)
    date_updated = models.DateTimeField(auto_now=True)
    
    class Meta:
        verbose_name = 'Partner Organization Feedback & Support'
        verbose_name_plural = 'Partner Organization Feedback & Supports'
    
    def __str__(self):
        return f"Feedback & Support for {self.partnering_organization.name} - {self.date_created.strftime('%Y-%m-%d')}"