from django.db import models
from django.dispatch import receiver
from django.utils.text import slugify
from django.db.models.signals import pre_save
from django.core.exceptions import ValidationError

from .UserProfile import UserProfile

# PARTICIPANT MODEL
class Participant(models.Model):
    SEX_CHOICES = [
        ('Male', 'Male'),
        ('Female', 'Female'),
    ]
    name = models.CharField(max_length=100)
    phone = models.CharField(max_length=15, blank=True, null=True, unique=False)
    email = models.EmailField(blank=True, null=True, unique=False)
    age = models.IntegerField(blank=True, null=True, unique=False)
    sex = models.CharField(max_length=7, choices=SEX_CHOICES, blank=True, null=True)
    slug = models.SlugField(max_length=120, unique=True, blank=True)
    def save(self, *args, **kwargs):
        # Set default values for phone and email if they are empty
        if not self.phone:
            self.phone = '000000000'
        if not self.email:
            self.email = 'No@email.com'
        # Generate a unique slug
        if not self.slug:
            base_slug = slugify(self.name)
            slug = base_slug
            count = 1
            while Participant.objects.filter(slug=slug).exists():
                slug = f"{base_slug}-{count}"
                count += 1
            self.slug = slug
        super().save(*args, **kwargs)
    def __str__(self):
        return self.name
    class Meta:
        verbose_name = 'Camp Participant'
        verbose_name_plural = 'Camp Participants'
    
# WORKSHOP MODEL
class WorkshopChoice(models.Model):
    workshop_title = models.CharField(max_length=100, unique=True)
    workshop_officer = models.ForeignKey(UserProfile, on_delete=models.SET_NULL, null=True, limit_choices_to={'is_moderator': True}, related_name='workshops', help_text='Only users who are workshop moderators can be assigned here.')
    workshop_description = models.CharField(max_length=100, unique=True)
    

    def __str__(self):
        return f"Workshop: {self.workshop_title} By: (Officer: {self.workshop_officer})"
    
    class Meta:
        verbose_name = 'Camp Workshop Choice'
        verbose_name_plural = 'Camp Workshop Choices'

    def clean(self):
        if self.workshop_officer and not self.workshop_officer.is_moderator:
            raise ValidationError('The selected user is not a workshop moderator.')

class ParticipantWorkshop(models.Model):
    participant_count = models.CharField(max_length=20, default='0')
    workshop = models.ForeignKey(WorkshopChoice, on_delete=models.CASCADE, null=True,)
    lessons_learned = models.TextField(blank=True, null=True)

    def __str__(self):
        return f"{self.participant_count} attended {self.workshop.workshop_title}"
    
    class Meta:
        verbose_name = 'Camp Participant Workshop'
        verbose_name_plural = 'Camp Participant Workshops'

# SERVICE MODEL
class ServiceChoice(models.Model):
    service_title = models.CharField(max_length=100, unique=True)
    service_description = models.TextField(blank=True, null=True)
    service_officer = models.ForeignKey(UserProfile, on_delete=models.SET_NULL, null=True,limit_choices_to={'is_nurse': True} | {'is_doctor': True}, related_name='services', help_text='Only users who are nurses or doctors can be assigned here.')
    
    slug = models.SlugField(max_length=120, unique=True, blank=True)
    def save(self, *args, **kwargs):
        if not self.slug:
            self.slug = slugify(self.service_title)
        super().save(*args, **kwargs)
        
    def __str__(self):
        return f"Service: {self.service_title} By: (Officer: {self.service_officer})"
    
    class Meta:
        verbose_name = 'Camp Service Choice'
        verbose_name_plural = 'Camp Service Choices'

    def clean(self):
        if self.service_officer and not (self.service_officer.is_nurse or self.service_officer.is_doctor):
            raise ValidationError('The selected user must be either a nurse or a doctor.')

class HealthService(models.Model):
    participant = models.ForeignKey('Participant', on_delete=models.CASCADE, related_name='health_services')
    service = models.ForeignKey(ServiceChoice, on_delete=models.CASCADE, null=True, related_name='health_service')
    participant_results = models.TextField(blank=True, null=True)  
    service_officer_officer_feedback = models.TextField(blank=True, null=True)
    doctor = models.ForeignKey(UserProfile, on_delete=models.SET_NULL, null=True, limit_choices_to={'is_doctor': True}, related_name='health_service_doctor', help_text='Only users who are doctors can be assigned here.')
    doctors_prescription = models.TextField(blank=True, null=True)
    pharmacist_prescription = models.TextField(blank=True, null=True)
    doctors_feedback = models.TextField(blank=True, null=True)
    pharmacist_feedback = models.TextField(blank=True, null=True)
    participated = models.BooleanField(default=False)
    prescribed = models.BooleanField(default=False)
    medicated = models.BooleanField(default=False)

    def __str__(self):
        return f"{self.participant.name} has received {self.service.service_title}"
    
    class Meta:
        verbose_name = 'Camp Health Service'
        verbose_name_plural = 'Camp Health Service'



# FEEDBACK MODEL
class Feedback(models.Model):
    participant = models.ForeignKey(Participant, on_delete=models.CASCADE, related_name='feedbacks')
    feedback = models.TextField()
    lessons_learned = models.TextField() 

    def __str__(self):
        return f"Feedback from {self.participant.name}"
    
    class Meta:
        verbose_name = 'Camp Feedback'
        verbose_name_plural = 'Camp Feedback'


# TREEPLANTING MODEL
class TreePlanting(models.Model):
    number_of_trees = models.PositiveIntegerField()
    date_planted = models.DateField()
    start_date_of_monitoring = models.DateField(null=True, blank=True)
    end_date_of_monitoring = models.DateField(null=True, blank=True)
    progress_update = models.TextField()  

    def __str__(self):
        return f"{self.number_of_trees} trees planted on {self.date_planted}"
    
    class Meta:
        verbose_name = 'Camp Tree Planting'
        verbose_name_plural = 'Camp Tree Planting'
