
import csv
import pdfkit
import datetime
from django.contrib import admin
from django.utils import timezone
from django.http import HttpResponse
from django.utils.html import format_html
from django.utils.safestring import mark_safe
from django.core.exceptions import ValidationError


# from .models import *
from .models.Note import Note
from .models.WebData import WebData
from .models.calendar import Calendar
from .models.UserFiles import UserFile
from .models.CampItems import CampItem
from .models.Volunteer import Volunteer
from .models.Minutes import Minutes
from .models.UserProfile import UserProfile
from .models.ActivityLog import ActivityLog
from .models.PrivacyPolicy import PrivacyPolicy
from .models.ProgressReport import ProgressReport
from .models.OrganizationFiles import OrganizationFile 
from .models.OrganizationType import OrganizationType
from .models.VolunteerAgreement import VolunteerAgreement
from .models.TermsAndConditions import TermsAndConditions
from .models.PartnershipAgreement import PartnershipAgreement
from .models.AdminNotifications import AdminNotifications,NotificationReadBy
from .models.ContactAdminMessages import ContactAdminMessages,ContactDeveloper
from .models.PartneringOrganization import PartneringOrganization, PartnerOrganizationFeedbackAndSupport
from .models.Projects import Project, Beneficiary, CoreProblem, SolutionTree, Milestone, Result, Activity, Resource, Stakeholder, Indicator, Beneficiary
from .models.SupportDocuments import SupportDocument
from .models.HealthCamp import Participant, WorkshopChoice, ParticipantWorkshop, ServiceChoice, HealthService, Feedback, TreePlanting

# webdata = WebData.objects.first()

# if webdata:
#     site_title = webdata.title
# else:
#     site_title = "Admin Dashboard" 

# admin.site.site_header = f"{site_title} Admin Dashboard"
# admin.site.site_title = f"{site_title} Admin Portal"
# admin.site.index_title = f"Welcome to the {site_title} Admin Portal"

# Register your models here.
admin.site.register(Participant)
admin.site.register(WorkshopChoice)
admin.site.register(ParticipantWorkshop)
admin.site.register(ServiceChoice)
admin.site.register(HealthService)
admin.site.register(Feedback)
admin.site.register(TreePlanting)


@admin.register(PartnerOrganizationFeedbackAndSupport)
class PartnerOrganizationFeedbackAndSupportAdmin(admin.ModelAdmin):
    list_display = ('partnering_organization', 'date_created', 'date_updated')
    search_fields = ('partnering_organization__name',)
    list_filter = ('date_created',)


@admin.register(CampItem)
class CampItemAdmin(admin.ModelAdmin):
    list_display = ('camp_item_photo','item_description', 'volunteer', 'partnering_organization')
    search_fields = ('item_description',)
    list_filter = ('volunteer', 'partnering_organization')
    
    def camp_item_photo(self, obj):
        if obj.items:
            return format_html('<img src="{}" width="50" height="50" style="border-radius:50%;" />', obj.items.url)
        else:
            return "No Camp Photo"
    camp_item_photo.short_description = 'Camp Photo'

# contact model
class ContactDeveloperAdmin(admin.ModelAdmin):
    list_display = ('user_profile_photo', 'user_name', 'subject', 'request_date', 'seen', 'seen_date')
    
    def user_profile_photo(self, obj):
        if obj.user.profile_picture:
            return format_html('<img src="{}" width="50" height="50" style="border-radius:50%;" />', obj.user.profile_picture.url)
        else:
            return "No Photo"
    user_profile_photo.short_description = 'Profile Photo'

    def user_name(self, obj):
        return f"{obj.user.user.first_name} {obj.user.user.last_name}"
    user_name.short_description = 'User Name'


admin.site.register(ContactDeveloper, ContactDeveloperAdmin)


class ContactAdminMessagesAdmin(admin.ModelAdmin):
    list_display = ['name', 'email', 'phone', 'message', 'request_date', 'seen']
    list_filter = ['seen']
    actions = ['mark_as_seen', 'mark_as_not_seen']

    def mark_as_seen(self, request, queryset):
        queryset.update(seen=True)

    def mark_as_not_seen(self, request, queryset):
        queryset.update(seen=False)

    mark_as_seen.short_description = "Mark selected messages as seen"
    mark_as_not_seen.short_description = "Mark selected messages as not seen"

admin.site.register(ContactAdminMessages, ContactAdminMessagesAdmin)


# webdata model

@admin.register(WebData)
class WebDataAdmin(admin.ModelAdmin):
    list_display = ('title','mission', 'vision', 'goals')
    search_fields = ('title', )


# Organization Type model
class OrganizationTypeAdmin(admin.ModelAdmin):
    list_display = ('acronym','name')
    search_fields = ('name',)

admin.site.register(OrganizationType, OrganizationTypeAdmin)



# User model
@admin.register(UserProfile)
class UserProfileAdmin(admin.ModelAdmin):
    list_display = ('first_name','last_name','get_profile_picture', 'email')
    list_filter = ('is_active', 'is_verified')
    search_fields = ('username','email', 'phone')

    def get_profile_picture(self, obj):
        if obj.profile_picture:
            return mark_safe(f'<img src="{obj.profile_picture.url}" width="50" height="50" style="border-radius:50%"/>')
        else:
            return None
    get_profile_picture.short_description = 'Profile Picture'
    
    def first_name(self, obj):
            return obj.user.first_name
    first_name.short_description = 'User First name'
    
    def last_name(self, obj):
            return obj.user.last_name
    last_name.short_description = 'User Last name'
    def email(self, obj):
            return obj.user.email
    email.short_description = 'User Email'
    
    



@admin.register(CoreProblem)
class CoreProblemAdmin(admin.ModelAdmin):
    list_display = ('name', 'driver')
    search_fields = ('name', 'description', 'driver', 'root_causes', 'effects', 'impact')


@admin.register(SolutionTree)
class SolutionTreeAdmin(admin.ModelAdmin):
    list_display = ('core_problem',)
    list_filter = ('core_problem',)
    search_fields = ('core_problem__name', 'identified_solution', 'solution_drivers', 'inputs', 'activities', 'outputs', 'outcomes', 'intermediate_states', 'impact', 'results')





@admin.register(PartneringOrganization)
class PartneringOrganizationAdmin(admin.ModelAdmin):
    list_display = ('name','get_logo', 'website_link')
    search_fields = ('name', 'description')
    
    def get_logo(self, obj):
        if obj.logo:
            return mark_safe(f'<img src="{obj.logo.url}" width="50" height="auto"/>')
        else:
            return None
    get_logo.short_description = 'Logo'
    
    
@admin.register(Project)
class ProjectAdmin(admin.ModelAdmin):
    list_display = ('title', 'author', 'category', 'donation_needed', 'donation_raised', 'donation_remaining', 'is_active')
    list_filter = ('category', 'is_active')
    search_fields = ('title', 'author__username')
    

    def donation_remaining(self, obj):
        return obj.donation_remaining

    donation_remaining.short_description = 'Donation Remaining'
    
    



@admin.register(ActivityLog)
class ActivityLogAdmin(admin.ModelAdmin):
    list_display = ('details', 'action','timestamp', 'ip_address', 'status_code','user_agent')
    list_filter = ('action', 'timestamp', 'status_code')
    search_fields = ('action', 'ip_address', 'user_agent')
    readonly_fields = ('timestamp',)

    def has_add_permission(self, request):
        return True  # Prevents adding new ActivityLog entries via admin

    def has_delete_permission(self, request, obj=None):
        return True  # Prevents deleting ActivityLog entries via admin
    
    
class NotificationReadByInline(admin.TabularInline):
    model = NotificationReadBy

@admin.register(AdminNotifications)
class AdminNotificationsAdmin(admin.ModelAdmin):
    list_display = ['message', 'created_at']
    list_filter = ['created_at']
    search_fields = ['message']
    inlines = [NotificationReadByInline]

@admin.register(NotificationReadBy)
class NotificationReadByAdmin(admin.ModelAdmin):
    list_display = ['notification', 'user', 'is_read']
    list_filter = ['is_read']
    search_fields = ['notification__message', 'user__username']
    



class VolunteerAdmin(admin.ModelAdmin):
    list_display = (
        'name', 'email', 'phone', 'volunteer_type', 'start_date', 'end_date', 'is_active'
    )
    search_fields = ('volunteer_type__name', 'name', 'email', 'phone')
    list_filter = ('volunteer_type', 'start_date', 'is_active')
    actions = ['export_as_csv', 'export_as_pdf']

    def export_as_csv(self, request, queryset):
        response = HttpResponse(content_type='text/csv')
        response['Content-Disposition'] = 'attachment; filename=volunteers.csv'
        writer = csv.writer(response)
        writer.writerow(['Name', 'Email', 'Phone', 'Volunteer Type', 'Start Date', 'End Date', 'Is Active'])
        for volunteer in queryset:
            writer.writerow([
                volunteer.name, volunteer.email, volunteer.phone, volunteer.volunteer_type.name,
                volunteer.start_date, volunteer.end_date, volunteer.is_active
            ])
        return response
    export_as_csv.short_description = "Export Selected as CSV"

    def export_as_pdf(self, request, queryset):
        response = HttpResponse(content_type='application/pdf')
        response['Content-Disposition'] = 'attachment; filename=volunteers.pdf'
        
        html = "<h1>Volunteer List</h1><table><tr><th>Name</th><th>Email</th><th>Volunteer Type</th><th>Start Date</th><th>End Date</th><th>Is Active</th></tr>"
        for volunteer in queryset:
            html += f"<tr><td>{volunteer.name}</td><td>{volunteer.email}</td><td>{volunteer.volunteer_type.name}</td><td>{volunteer.start_date}</td><td>{volunteer.end_date}</td><td>{volunteer.is_active}</td></tr>"
        html += "</table>"
        
        pdf_content = pdfkit.from_string(html, False)
        response.write(pdf_content)
        return response
    export_as_pdf.short_description = "Export Selected as PDF"

admin.site.register(Volunteer, VolunteerAdmin)

class PrivacyPolicyAdmin(admin.ModelAdmin):
    list_display = ['title', 'view_document']

    def view_document(self, obj):
        return mark_safe(f'<a href="{obj.document.url}" target="_blank">View Document</a>')

    view_document.short_description = 'Document'

admin.site.register(PrivacyPolicy, PrivacyPolicyAdmin)

class VolunteerAgreementAdmin(admin.ModelAdmin):
    list_display = ['title', 'view_document']

    def view_document(self, obj):
        return mark_safe(f'<a href="{obj.document.url}" target="_blank">View Document</a>')

    view_document.short_description = 'Document'

admin.site.register(VolunteerAgreement, VolunteerAgreementAdmin)

class PartnershipAgreementAdmin(admin.ModelAdmin):
    list_display = ['title', 'view_document']

    def view_document(self, obj):
        return mark_safe(f'<a href="{obj.document.url}" target="_blank">View Document</a>')

    view_document.short_description = 'Document'

admin.site.register(PartnershipAgreement, PartnershipAgreementAdmin)

class TermsAndConditionsAdmin(admin.ModelAdmin):
    list_display = ['title', 'view_document']

    def view_document(self, obj):
        return mark_safe(f'<a href="{obj.document.url}" target="_blank">View Document</a>')

    view_document.short_description = 'Document'

admin.site.register(TermsAndConditions, TermsAndConditionsAdmin)


class NoteAdmin(admin.ModelAdmin):
    list_display = ('title', 'user', 'date', 'tag', 'isFav')
    list_filter = ('tag', 'isFav', 'date')
    search_fields = ('title', 'user__user__first_name', 'user__user__last_name')
    readonly_fields = ('thumb', 'date')
    fieldsets = (
        (None, {
            'fields': ('title', 'user', 'description', 'date', 'isFav', 'tag')
        }),
        ('Additional Information', {
            'fields': ('thumb',),
            'classes': ('collapse',),
        }),
    )
    ordering = ('-date',)

admin.site.register(Note, NoteAdmin)



class ProgressReportAdmin(admin.ModelAdmin):
    list_display = ('title', 'user_profile', 'progress_report_priority', 'progress_report_tags')
    search_fields = ('title', 'user_profile__user__first_name', 'user_profile__user__last_name', 'progress_report_priority', 'progress_report_tags')
    list_filter = ('progress_report_priority', 'progress_report_tags',)
    filter_horizontal = ('seen_by', 'progress_report_document')

admin.site.register(ProgressReport, ProgressReportAdmin)




@admin.register(Calendar)
class CalendarAdmin(admin.ModelAdmin):
    list_display = ('title', 'start', 'end', 'color_code', 'badge', 'updated_at')
    list_filter = ('color_code', 'badge', 'start', 'end')
    search_fields = ('title', 'description')
    ordering = ('-start',)

@admin.register(Milestone)
class MilestoneAdmin(admin.ModelAdmin):
    list_display = ('name', 'start_date', 'end_date', 'project', 'is_achieved', 'date_of_achievement', 'cost')
    list_filter = ('is_achieved', 'project', 'start_date', 'end_date')
    search_fields = ('name', 'project__title')
    ordering = ('-start_date',)

@admin.register(Result)
class ResultAdmin(admin.ModelAdmin):
    list_display = ('project', 'output', 'outcome', 'results_achieved', 'is_achieved', 'global_contribution')
    list_filter = ('is_achieved', 'project')
    search_fields = ('project__title', 'output', 'outcome')
    ordering = ('-project',)

@admin.register(Activity)
class ActivityAdmin(admin.ModelAdmin):
    list_display = ('name', 'status', 'start_date', 'end_date', 'user_responsible', 'cost')
    list_filter = ('status', 'start_date', 'end_date', 'user_responsible')
    search_fields = ('name', 'goals_and_objectives', 'user_responsible__username')
    ordering = ('-start_date',)

@admin.register(Resource)
class ResourceAdmin(admin.ModelAdmin):
    list_display = ('name', 'number', 'unit_cost', 'total_cost', 'use_function')
    list_filter = ('name',)
    search_fields = ('name', 'use_function')
    ordering = ('name',)

@admin.register(Stakeholder)
class StakeholderAdmin(admin.ModelAdmin):
    list_display = ('name', 'location', 'affect_project', 'affected_by_project', 'gain_from_project')
    list_filter = ('location', 'affect_project', 'affected_by_project')
    search_fields = ('name', 'location', 'organization__name')
    ordering = ('-created_at',)

@admin.register(Indicator)
class IndicatorAdmin(admin.ModelAdmin):
    list_display = ('name', 'measure_type', 'measure_unit', 'baseline', 'target', 'relevance')
    list_filter = ('measure_type', 'measure_unit')
    search_fields = ('name', 'description')
    ordering = ('name',)

@admin.register(Beneficiary)
class BeneficiaryAdmin(admin.ModelAdmin):
    list_display = ('name', 'location', 'category', 'priority', 'strength', 'weakness')
    list_filter = ('category', 'priority', 'created_at')
    search_fields = ('name', 'location', 'description')
    ordering = ('-created_at',)


# Register the Minutes model
@admin.register(Minutes)
class MinutesAdmin(admin.ModelAdmin):
    list_display = ('meeting_title', 'meeting_date', 'created_by', 'created_at', 'updated_at')
    search_fields = ('meeting_title', 'attendees')
    list_filter = ('meeting_date', 'created_by')
    prepopulated_fields = {'slug': ('meeting_title', 'meeting_date',)}
    readonly_fields = ('created_at', 'updated_at')

@admin.register(OrganizationFile)
class OrganizationFileAdmin(admin.ModelAdmin):
    list_display = ('name', 'created_at', 'updated_at')
    search_fields = ('name',)
    prepopulated_fields = {'slug': ('name',)}
    readonly_fields = ('created_at', 'updated_at')

@admin.register(UserFile)
class UserFileAdmin(admin.ModelAdmin):
    list_display = ('name', 'user', 'upload_date', 'updated_at')
    search_fields = ('name', 'user__username')
    list_filter = ('user', 'upload_date')
    prepopulated_fields = {'slug': ('name',)}
    readonly_fields = ('upload_date', 'updated_at')